// -*- mode:C++; tab-width:8; c-basic-offset:2; indent-tabs-mode:t -*-
// vim: ts=8 sw=2 smarttab

#include "journal/JournalMetadata.h"
#include "test/journal/RadosTestFixture.h"
#include "common/Cond.h"
#include "common/Mutex.h"
#include <map>

class TestJournalMetadata : public RadosTestFixture {
public:
  virtual void TearDown() {
    for (MetadataList::iterator it = m_metadata_list.begin();
         it != m_metadata_list.end(); ++it) {
      (*it)->remove_listener(&m_listener);
    }
    m_metadata_list.clear();

    RadosTestFixture::TearDown();
  }

  journal::JournalMetadataPtr create_metadata(const std::string &oid,
                                              const std::string &client_id,
                                              double commit_internal = 0.1) {
    journal::JournalMetadataPtr metadata = RadosTestFixture::create_metadata(
      oid, client_id, commit_internal);
    m_metadata_list.push_back(metadata);
    metadata->add_listener(&m_listener);
    return metadata;
  }

  typedef std::list<journal::JournalMetadataPtr> MetadataList;
  MetadataList m_metadata_list;
};

TEST_F(TestJournalMetadata, JournalDNE) {
  std::string oid = get_temp_oid();

  journal::JournalMetadataPtr metadata1 = create_metadata(oid, "client1");
  ASSERT_EQ(-ENOENT, init_metadata(metadata1));
}

TEST_F(TestJournalMetadata, ClientDNE) {
  std::string oid = get_temp_oid();

  ASSERT_EQ(0, create(oid, 14, 2));
  ASSERT_EQ(0, client_register(oid, "client1", ""));

  journal::JournalMetadataPtr metadata1 = create_metadata(oid, "client1");
  ASSERT_EQ(0, init_metadata(metadata1));

  journal::JournalMetadataPtr metadata2 = create_metadata(oid, "client2");
  ASSERT_EQ(-ENOENT, init_metadata(metadata2));
}

TEST_F(TestJournalMetadata, Committed) {
  std::string oid = get_temp_oid();

  ASSERT_EQ(0, create(oid, 14, 2));
  ASSERT_EQ(0, client_register(oid, "client1", ""));

  journal::JournalMetadataPtr metadata1 = create_metadata(oid, "client1", 600);
  ASSERT_EQ(0, init_metadata(metadata1));

  journal::JournalMetadataPtr metadata2 = create_metadata(oid, "client1");
  ASSERT_EQ(0, init_metadata(metadata2));
  ASSERT_TRUE(wait_for_update(metadata2));

  journal::JournalMetadata::ObjectSetPosition expect_commit_position;
  journal::JournalMetadata::ObjectSetPosition read_commit_position;
  metadata1->get_commit_position(&read_commit_position);
  ASSERT_EQ(expect_commit_position, read_commit_position);

  uint64_t commit_tid1 = metadata1->allocate_commit_tid(0, 0, 0);
  uint64_t commit_tid2 = metadata1->allocate_commit_tid(0, 1, 0);
  uint64_t commit_tid3 = metadata1->allocate_commit_tid(1, 0, 1);
  uint64_t commit_tid4 = metadata1->allocate_commit_tid(0, 0, 2);

  // cannot commit until tid1 + 2 committed
  metadata1->committed(commit_tid2, []() { return nullptr; });
  metadata1->committed(commit_tid3, []() { return nullptr; });

  C_SaferCond cond1;
  metadata1->committed(commit_tid1, [&cond1]() { return &cond1; });

  // given our 10 minute commit internal, this should override the
  // in-flight commit
  C_SaferCond cond2;
  metadata1->committed(commit_tid4, [&cond2]() { return &cond2; });

  ASSERT_EQ(-ESTALE, cond1.wait());
  metadata1->flush_commit_position();
  ASSERT_EQ(0, cond2.wait());

  ASSERT_TRUE(wait_for_update(metadata2));
  metadata2->get_commit_position(&read_commit_position);
  expect_commit_position = {{{0, 0, 2}, {1, 0, 1}}};
  ASSERT_EQ(expect_commit_position, read_commit_position);
}

TEST_F(TestJournalMetadata, UpdateActiveObject) {
  std::string oid = get_temp_oid();

  ASSERT_EQ(0, create(oid, 14, 2));
  ASSERT_EQ(0, client_register(oid, "client1", ""));

  journal::JournalMetadataPtr metadata1 = create_metadata(oid, "client1");
  ASSERT_EQ(0, init_metadata(metadata1));
  ASSERT_TRUE(wait_for_update(metadata1));

  ASSERT_EQ(0U, metadata1->get_active_set());

  metadata1->set_active_set(123);
  ASSERT_TRUE(wait_for_update(metadata1));

  ASSERT_EQ(123U, metadata1->get_active_set());
}
